import { defineConfig } from 'vite'
import path from 'path'
import child_process from 'child_process'
import fs from 'fs-extra'
import banner from 'vite-plugin-banner'
import react from '@vitejs/plugin-react-swc'
import cssInjectedByJsPlugin from 'vite-plugin-css-injected-by-js'
import commonjs from 'vite-plugin-commonjs'
import pkg from './package.json'

// import requireTransform from "vite-plugin-require-transform"
const rootDir = path.resolve(__dirname)

const commitid = child_process.execSync('git show -s --format=%H').toString().trim().slice(0, 8)

const branch = child_process.execSync('git rev-parse --abbrev-ref HEAD').toString().trim()

export const requireTransform = function () {
  const copyList: string[][] = []

  return {
    name: 'requireTransform',
    transform(code: string, id: string) {
      if (/node_modules/.test(id)) {
        return code
      }

      const regex = /require\(["'](.+?\.(?:png|jpe?g|gif|bmp|svg))["']\)/gi
      const matchs = code.matchAll(regex)
      if (!matchs) {
        return code
      }

      let str = code

      const dir = path.dirname(id)

      ;[...matchs].forEach(async (match) => {
        const file_path = path.resolve(dir, match[1])

        const stats = fs.statSync(file_path)
        const fileSizeInBytes = stats.size

        if (fileSizeInBytes > 10240) {
          str = str.replaceAll(match[0], `{uri: "${match[1]}"}`)
          copyList.push([file_path, match[1]])
        }
      })

      return str
    },
    writeBundle: {
      sequential: true,
      order: 'post',
      async handler(target: any) {
        copyList.map(async (item) => {
          const new_path = path.resolve(target.dir, item[1])
          const target_dir = path.dirname(new_path)
          await fs.ensureDir(target_dir)
          await fs.copy(item[0], new_path)
        })
      }
    }
  } as any
}

// https://vitejs.dev/config/
export default defineConfig({
  mode: 'production',
  plugins: [
    react(),
    requireTransform(),
    commonjs(),
    cssInjectedByJsPlugin(),
    banner(`This version is build from branch: ${branch} , commiid: ${commitid}, and package version is ${pkg.version}`)
  ],
  define: {
    'process.env.NODE_ENV': JSON.stringify('production')
  },
  build: {
    minify: 'terser',
    lib: {
      entry: path.resolve(__dirname, 'index.ts'),
      formats: ['iife'],
      name: 'test'
    },
    rollupOptions: {
      external: ['react', 'react-native'],
      output: {
        entryFileNames: `${pkg.name}.min.v${pkg.version}.js`,
        globals: {
          react: '__widget_deps.rt',
          'react-native': '__widget_deps.rnw'
        }
      }
    },
    assetsInlineLimit: 4090
  },
  assetsInclude: ['**/*.png'],
  resolve: {
    alias: [
      {
        find: '@',
        replacement: path.resolve(__dirname, './src')
      },
      {
        find: 'react-native',
        replacement: path.resolve('./', '../../node_modules/react-native-web')
      }
    ]
  }
})
