const fs = require('fs')
const archiver = require('archiver')
const path = require('path')
const pkg = require('../package.json')
const { spawn } = require('child_process')

const distPath = path.join(__dirname, '../dist')

const commonTasks = ['common-android', 'common-ios']
const mobileTasks = ['android', 'ios']
// 打包顺序要保证基础包 -> 业务包
const tasks = ['common-android', 'android', 'common-ios', 'ios', 'web']

const command = 'yarn'

function execCommand(command, args, cb) {
  const childProcess = spawn(command, args, { shell: process.platform === 'win32' })

  childProcess.stdout.on('data', (data) => {
    console.log(String(data))
  })

  childProcess.stderr.on('data', (data) => {
    console.error(String(data))
  })

  childProcess.on('close', (code) => {
    cb && cb(code)
  })
}

function deleteFolderRecursive(folderPath, deleteTopFolder) {
  if (fs.existsSync(folderPath)) {
    fs.readdirSync(folderPath).forEach((file, index) => {
      const curPath = path.join(folderPath, file)
      if (fs.lstatSync(curPath).isDirectory()) {
        deleteFolderRecursive(curPath, true)
      } else {
        fs.unlinkSync(curPath)
      }
    })
    deleteTopFolder && fs.rmdirSync(folderPath)
  }
}

function zipAndDestroy(task) {
  // 打包并且清空目录
  return new Promise((resolve, reject) => {
    const isCommonTask = commonTasks.includes(task)
    const inputFolder = path.resolve(__dirname, '../dist')
    const outputZip = `./bundle/${isCommonTask ? 'common/' : ''}${task}-v${pkg.version}.zip`

    const output = fs.createWriteStream(outputZip)
    const archive = archiver('zip')

    output.on('error', function (err) {
      reject(err)
    })

    output.on('close', function () {
      console.log(`${task}-${pkg.version}版本的压缩包已创建: ${outputZip}`)
      resolve()
      deleteFolderRecursive(distPath)
    })

    archive.on('warning', function (err) {
      if (err.code === 'ENOENT') {
        // log warning
        console.log(err)
      } else {
        // throw error
        throw err
      }
    })

    // 绑定压缩流到输出流
    archive.pipe(output)

    // 将打包后的文件夹添加到压缩包中
    archive.directory(inputFolder, false)

    // 完成并关闭压缩包
    archive.finalize()
  })
}

;(async () => {
  const outputDir = path.resolve('__dirname', '../bundle/common/')
  fs.mkdirSync(distPath, { recursive: true })
  fs.mkdirSync(outputDir, { recursive: true })
  for (task of tasks) {
    deleteFolderRecursive(distPath)

    await new Promise((resolve, reject) => {
      execCommand(command, [`build:${task}`], async (err) => {
        if (!err) {
          if ([...commonTasks, ...mobileTasks].includes(task)) {
            await zipAndDestroy(task)
          } else {
            const filename = `${pkg.name}.min.v${pkg.version}.js`
            fs.rename(path.resolve(distPath, filename), path.resolve(distPath, `../bundle/${filename}`), (err) => {})
          }

          console.log(`${task}-${pkg.version}构建成功！`)
          resolve()
        } else {
          reject()
        }
      })
    })
  }
})()
